# Scoring Animations Refactoring Guide

**Date:** October 14, 2025
**Version:** 2.0
**Backup Location:** `Originals_Backup_20251014_224126/`

## Summary of Changes

This refactoring standardizes all scoring animation scripts to use a common library, consistent naming conventions, and ESPN team IDs in output paths.

---

## Key Changes

### 1. **CSV Data Source**
- **Old:** `teams_rgb.csv` (missing espn_team_id column, inconsistent structure)
- **New:** `teams_rgb_correct.csv` (includes espn_team_id, location, and other metadata)

**New CSV Structure:**
```csv
"id","league","conference","abbreviation","displayName","location","R","G","B","R2","G2","B2","espn_team_id"
"1","NFL",NULL,"ARI","Cardinals","Arizona","164","2","39","255","255","255","22"
```

### 2. **Common Library (`gl_common.jsxinc`)**
All JSX scripts now use the shared `gl_common.jsxinc` library which provides:
- CSV parsing with espn_team_id support
- Smart color computation (handles black colors gracefully)
- Consistent logo replacement
- Standardized output path generation
- Render queue management
- Text/solid/tint color helpers

**Usage in JSX scripts:**
```javascript
// Load library
(function(){
  var COMMON = $.getenv("AE_COMMON_JSX") || "";
  if (COMMON) { $.evalFile(File(COMMON)); }
  if (!$.global.GL) { throw new Error("Common library not loaded."); }
})();
var GL = $.global.GL;

// Use GL functions
var teams = GL.buildTeams(rows, LEAGUE);
var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
```

### 3. **Output Path Structure**
- **Old:** `{LEAGUE}_{CONFERENCE}_{ABBR}_{ANIM}/`
- **New:** `{LEAGUE}_{ESPN_TEAM_ID}_{ABBR}_{ANIM}/`

**Example:**
- Old: `TD_Animation_3_384/NFL/NFL_AFC_BUF_TD_2_6/`
- New: `TD_Animation_3_384/NFL/NFL_2_BUF_TD_2_6/`  (where `2` is Buffalo's ESPN ID)

**Benefits:**
- ESPN team IDs are stable and match external systems
- Conference data is often missing or inconsistent
- Easier integration with APIs and databases

### 4. **Python Runner Standardization**
All Python runners now:
- Reference `teams_rgb_correct.csv`
- Set `AE_COMMON_JSX` environment variable
- Follow consistent structure from `_python_runner_template.py`
- Include validation for all required files

---

## File Naming Conventions

### After Effects Projects (.aep)
**Format:** `{Sport}_{ScoreType}_Animation{Number}_{Resolution}.aep`

**Examples:**
- `Football_TD_Animation1_192.aep`
- `Football_FG_Animation3_384.aep`
- `Baseball_HomeRun_Animation2_192.aep`
- `Hockey_Goal_Animation1_192.aep`

### JSX Batch Scripts (.jsx)
**Format:** `batch_{Sport}_{ScoreType}_Animation{Number}_{Resolution}.jsx`

**Examples:**
- `batch_Football_TD_Animation1_192.jsx`
- `batch_Baseball_HomeRun_Animation3_192.jsx`

### Python Runners (.py)
**Format:** `run_{Sport}_{ScoreType}_Animation{Number}_{Resolution}.py`

**Examples:**
- `run_Football_TD_Animation1_192.py`
- `run_Baseball_HomeRun_Animation2_192.py`

### Output Folders
**Format:** `{Sport}_{ScoreType}_Animation_{Number}_{Resolution}/`

**Examples:**
- `Football_TD_Animation_3_384/`
- `Baseball_HomeRun_Animation_1_192/`

---

## Animation Number Reference

### Football Animations
| Number | Type | Description | Resolution |
|--------|------|-------------|------------|
| 1 | TD | Basic touchdown animation | 192 |
| 2 | TD | Logo spin touchdown | 192/384 |
| 3 | TD | Stroke + solid recolor | 192/384 |
| 4 | TD | Spike animation | 192 |
| 1 | FG | Basic field goal | 192 |
| 2 | FG | Logo spin field goal | 192 |
| 3 | FG | Stroke + solid FG | 384 |
| 4 | FG | Advanced FG | 192 |
| 1-4 | 2PT | Two-point conversion variants | 192 |

### Baseball Animations
| Number | Type | Description | Resolution |
|--------|------|-------------|------------|
| 1 | HR | Basic home run | 192 |
| 2 | HR | Advanced home run | 192 |
| 3 | HR | Shatter home run | 192 |
| 1-3 | RS | Run scored variants | 192 |

### Hockey Animations
| Number | Type | Description | Resolution |
|--------|------|-------------|------------|
| 1 | GOAL | Break glass goal | 192 |

---

## Environment Variables Reference

### Required Variables
- `AE_PROJECT` - Path to .aep project file
- `AE_CSV` - Path to teams_rgb_correct.csv
- `AE_COMMON_JSX` - Path to gl_common.jsxinc
- `AE_COMP` - Composition name in After Effects
- `AE_LEAGUE` - Target league (NFL, MLB, NBA, NHL, CFB, etc.)
- `AE_OUTDIR` - Output directory path
- `AE_ANIM` - Animation identifier (e.g., "TD_2_6")

### Optional Variables
- `AE_TEAMS` - Comma-separated team abbreviations for subset mode
- `AE_LIMIT` - Limit number of teams processed
- `AE_PATH_TEMPLATE` - Output path template (default: "{league}")
- `AE_OM_TEMPLATE` - Output Module template (default: "PNG Sequence")
- `AE_EXT` - File extension for video outputs (e.g., ".mp4")
- `AE_PURGE_BEFORE_RENDER` - Purge caches before render ("1"|"0", default: "1")
- `AE_NO_RENDER` - Skip rendering ("1"|"0", default: "0")
- `AE_QUIT` - Quit After Effects when done ("1"|"0", default: "1")

### Layer Override Variables
- `AE_TEXT_STROKE_LAYER` - Main text layer name
- `AE_TEAMNAME_LAYER` - Team name layer
- `AE_LOGO_LAYER` - Logo layer name
- `AE_SOLID_LAYER` - Solid color layer

### Logo Variables
- `AE_LOGO_DIR` - Logo directory path
- `AE_LOGO_PATH_TEMPLATE` - Logo path template (default: "{league}/{abbr}")
- `AE_LOGO_EXTS` - Logo file extensions (default: "png,jpg,jpeg,svg,ai,psd")

---

## Migration Status

### ✅ Completed
1. Created timestamped backup (`Originals_Backup_20251014_224126/`)
2. Updated `gl_common.jsxinc` to support ESPN team IDs
3. Updated all GL-based JSX scripts to pass `espn_team_id` parameter
4. Updated all 28 Python runners to use `teams_rgb_correct.csv`
5. Created standardized Python runner template
6. Created this documentation

### ⚠️ Remaining Work
1. **Migrate non-GL JSX files** - The following files still contain inline duplicate code and should be refactored to use `gl_common.jsxinc`:
   - `batch_score_animation_3.jsx`
   - `batch_score_animation_3-pngOutput.jsx`
   - `batch_score_animation_3_fg.jsx`
   - `batch_score_animation_3_fg-pngOutput.jsx`
   - `batch_ScoringAnimations4.jsx`
   - `batch_ScoringAnimations4-CFB.jsx`
   - `batch_ScoringAnimations4-FG.jsx`
   - `batch_ScoringAnimations4-2PT.jsx`
   - `batch_ScoringAnimations4-2PT-CFB.jsx`
   - `batch_Baseball-Homerun-1.jsx`
   - `batch_Baseball-RunScored-1.jsx`

2. **Rename files** - Optional: rename .aep, .jsx, and .py files to match new naming convention

3. **Test end-to-end** - Run at least one animation for each sport to verify output paths and espn_team_id integration

---

## Testing Checklist

Before using updated scripts in production:

- [ ] Verify `teams_rgb_correct.csv` has all required teams with valid espn_team_ids
- [ ] Test one TD animation (Football)
- [ ] Test one FG animation (Football)
- [ ] Test one Home Run animation (Baseball)
- [ ] Test one Goal animation (Hockey)
- [ ] Verify output folder naming uses ESPN team ID instead of conference
- [ ] Confirm logos load correctly
- [ ] Verify team colors apply correctly
- [ ] Check that PNG sequences or MP4s render successfully

---

## Rollback Instructions

If you need to revert changes:

1. **Restore from backup:**
   ```bash
   cd "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations"
   cp Originals_Backup_20251014_224126/* ./
   ```

2. **Revert Python runners to old CSV:**
   ```bash
   # Manually edit Python files to change:
   # teams_rgb_correct.csv → teams_rgb.csv
   # And remove COMMON_JSX references
   ```

3. **Remove gl_common.jsxinc updates:**
   ```bash
   # Restore original gl_common.jsxinc from backup if needed
   ```

---

## Support & Questions

For questions about this refactoring:
1. Review this document
2. Check `_python_runner_template.py` for Python structure
3. Check `batch_ScoreAnimations1-TD.jsx` for JSX structure using GL library
4. Review `gl_common.jsxinc` for available helper functions

---

## Version History

### Version 2.0 (October 14, 2025)
- Migrated to ESPN team IDs in output paths
- Standardized all scripts to use `gl_common.jsxinc`
- Updated all Python runners to use `teams_rgb_correct.csv`
- Created comprehensive documentation

### Version 1.0 (Pre-refactoring)
- Individual JSX files with inline duplicate code
- Inconsistent output path structures
- Mixed use of conference vs team ID in paths
- No standardized Python runner template
